"use client";
import Link from "next/link";
import { useEffect, useState } from "react";
import { useSession } from "next-auth/react";
import { useRouter } from "next/router";
import { useLanguage } from "@/contexts/LanguageContext";
import {
  ArrowLeft,
  Filter,
  CheckCircle2,
  XCircle,
  Calendar,
  Building2,
  MapPin,
  Wrench,
  FileText,
  RefreshCw,
} from "lucide-react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";

type TaskItemCompletion = {
  id: string;
  elementId: string;
  status: "OK" | "NOT_OK";
  checkedAt: string;
  ticketId: string | null;
  notes: string | null;
  task: {
    id: string;
    title: string;
    scopeType: string | null;
    department: {
      id: string;
      name: string;
    } | null;
    location: {
      id: string;
      name: string;
    } | null;
    element: {
      id: string;
      name: string;
    } | null;
  };
  element: {
    id: string;
    name: string;
    location: {
      id: string;
      name: string;
      department: {
        id: string;
        name: string;
      };
    };
  };
  ticket: {
    id: string;
    title: string;
    status: string;
  } | null;
};

export default function TaskHistory() {
  const { data: session, status } = useSession();
  const router = useRouter();
  const { t } = useLanguage();

  const [completions, setCompletions] = useState<TaskItemCompletion[]>([]);
  const [loading, setLoading] = useState(true);
  const [err, setErr] = useState<string | null>(null);
  const [filtering, setFiltering] = useState(false);

  // Filters
  const [statusFilter, setStatusFilter] = useState<string>("all");
  const [dateFrom, setDateFrom] = useState<string>("");
  const [dateTo, setDateTo] = useState<string>("");
  const [hasTicketFilter, setHasTicketFilter] = useState<string>("all");

  const loadHistory = async () => {
    setLoading(true);
    setErr(null);
    try {
      const params = new URLSearchParams();
      if (statusFilter !== "all") {
        params.append("status", statusFilter);
      }
      if (dateFrom) {
        params.append("dateFrom", dateFrom);
      }
      if (dateTo) {
        params.append("dateTo", dateTo);
      }
      if (hasTicketFilter !== "all") {
        params.append("hasTicket", hasTicketFilter);
      }

      const res = await fetch(`/api/tasks/history?${params.toString()}`);
      const json = await res.json();
      if (json.ok) {
        setCompletions(json.completions || []);
      } else {
        setErr(json.error || "Failed to load history");
      }
    } catch (error: any) {
      setErr(error?.message || "Failed to load history");
    } finally {
      setLoading(false);
      setFiltering(false);
    }
  };

  useEffect(() => {
    if (status === "authenticated") {
      loadHistory();
    }
  }, [status]);

  const handleFilter = () => {
    setFiltering(true);
    loadHistory();
  };

  const handleClearFilters = () => {
    setStatusFilter("all");
    setDateFrom("");
    setDateTo("");
    setHasTicketFilter("all");
    setFiltering(true);
    setTimeout(() => {
      loadHistory();
    }, 100);
  };

  if (status === "unauthenticated") {
    return (
      <main className='max-w-4xl mx-auto px-4 py-6'>
        <p>{t("tasks.history.mustLogin") || "You must be logged in to view task history."}</p>
        <Link href='/login'>
          <Button>{t("common.logIn") || "Login"}</Button>
        </Link>
      </main>
    );
  }

  const role = (session?.user as any)?.role?.toUpperCase() || "";
  if (role !== "TECHNICIAN") {
    return (
      <main className='max-w-4xl mx-auto px-4 py-6'>
        <p>{t("tasks.history.onlyTechnician") || "Only technicians can view task history."}</p>
        <Link href='/tasks'>
          <Button variant='outline'>{t("tasks.history.backToTasks") || "Back to Tasks"}</Button>
        </Link>
      </main>
    );
  }

  return (
    <main className='max-w-6xl mx-auto px-4 py-6'>
      <div className='flex items-center justify-between mb-6'>
        <div className='flex items-center gap-3'>
          <Button variant='outline' asChild>
            <Link href='/tasks'>
              <ArrowLeft className='w-4 h-4 mr-1' />
              {t("tasks.history.backToTasks") || "Back to Tasks"}
            </Link>
          </Button>
          <h1 className='text-2xl font-bold'>{t("tasks.history.title") || "Task Completion History"}</h1>
        </div>
        <Button
          variant='outline'
          onClick={loadHistory}
          disabled={loading || filtering}
        >
          <RefreshCw
            className={`w-4 h-4 mr-1 ${
              loading || filtering ? "animate-spin" : ""
            }`}
          />
          {t("tasks.history.refresh") || "Refresh"}
        </Button>
      </div>

      {/* Filters */}
      <div className='bg-white shadow-sm rounded-xl border border-gray-100 p-4 mb-6'>
        <div className='flex items-center gap-2 mb-4'>
          <Filter className='w-4 h-4' />
          <h2 className='font-semibold'>{t("tasks.history.filters") || "Filters"}</h2>
        </div>
        <div className='grid grid-cols-1 md:grid-cols-4 gap-4'>
          <div className='space-y-2'>
            <Label htmlFor='status'>{t("tasks.history.status") || "Status"}</Label>
            <Select value={statusFilter} onValueChange={setStatusFilter}>
              <SelectTrigger id='status'>
                <SelectValue placeholder={t("tasks.history.allStatuses") || "All statuses"} />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value='all'>{t("tasks.history.allStatuses") || "All Statuses"}</SelectItem>
                <SelectItem value='OK'>{t("tasks.history.ok") || "OK"}</SelectItem>
                <SelectItem value='NOT_OK'>{t("tasks.history.notOk") || "NOT OK"}</SelectItem>
              </SelectContent>
            </Select>
          </div>

          <div className='space-y-2'>
            <Label htmlFor='dateFrom'>{t("tasks.history.dateFrom") || "Date From"}</Label>
            <Input
              id='dateFrom'
              type='date'
              value={dateFrom}
              onChange={(e) => setDateFrom(e.target.value)}
            />
          </div>

          <div className='space-y-2'>
            <Label htmlFor='dateTo'>{t("tasks.history.dateTo") || "Date To"}</Label>
            <Input
              id='dateTo'
              type='date'
              value={dateTo}
              onChange={(e) => setDateTo(e.target.value)}
            />
          </div>

          <div className='space-y-2'>
            <Label htmlFor='hasTicket'>{t("tasks.history.hasTicket") || "Has Ticket"}</Label>
            <Select value={hasTicketFilter} onValueChange={setHasTicketFilter}>
              <SelectTrigger id='hasTicket'>
                <SelectValue placeholder={t("tasks.history.all") || "All"} />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value='all'>{t("tasks.history.all") || "All"}</SelectItem>
                <SelectItem value='true'>{t("tasks.history.withTicket") || "With Ticket"}</SelectItem>
                <SelectItem value='false'>{t("tasks.history.withoutTicket") || "Without Ticket"}</SelectItem>
              </SelectContent>
            </Select>
          </div>
        </div>

        <div className='flex items-center gap-2 mt-4'>
          <Button onClick={handleFilter} disabled={filtering}>
            {t("tasks.history.applyFilters") || "Apply Filters"}
          </Button>
          <Button variant='outline' onClick={handleClearFilters}>
            {t("tasks.history.clearFilters") || "Clear Filters"}
          </Button>
        </div>
      </div>

      {/* Error Message */}
      {err && (
        <div className='bg-red-50 border border-red-200 rounded-lg p-4 mb-6'>
          <p className='text-red-800'>{err}</p>
        </div>
      )}

      {/* Results */}
      {loading ? (
        <div className='text-center py-12'>
          <RefreshCw className='w-8 h-8 mx-auto mb-2 animate-spin text-gray-400' />
          <p className='text-gray-500'>{t("tasks.history.loading") || "Loading history..."}</p>
        </div>
      ) : completions.length === 0 ? (
        <div className='text-center py-12 bg-white rounded-xl border border-gray-100'>
          <FileText className='w-12 h-12 mx-auto mb-3 text-gray-400' />
          <p className='text-gray-500'>{t("tasks.history.noHistory") || "No task completion history found."}</p>
        </div>
      ) : (
        <div className='bg-white shadow-sm rounded-xl border border-gray-100 overflow-hidden'>
          <div className='overflow-x-auto'>
            <table className='w-full'>
              <thead className='bg-gray-50 border-b'>
                <tr>
                  <th className='text-left p-3 font-semibold text-gray-700'>
                    {t("tasks.history.dateTime") || "Date & Time"}
                  </th>
                  <th className='text-left p-3 font-semibold text-gray-700'>
                    {t("tasks.title") || "Task"}
                  </th>
                  <th className='text-left p-3 font-semibold text-gray-700'>
                    {t("tasks.element") || "Element"}
                  </th>
                  <th className='text-left p-3 font-semibold text-gray-700'>
                    {t("tasks.location") || "Location"}
                  </th>
                  <th className='text-left p-3 font-semibold text-gray-700'>
                    {t("tasks.history.status") || "Status"}
                  </th>
                  <th className='text-left p-3 font-semibold text-gray-700'>
                    {t("tasks.detail.notes") || "Notes"}
                  </th>
                  <th className='text-left p-3 font-semibold text-gray-700'>
                    {t("tickets.title") || "Ticket"}
                  </th>
                </tr>
              </thead>
              <tbody>
                {completions.map((completion) => (
                  <tr
                    key={completion.id}
                    className='border-b hover:bg-gray-50 transition-colors'
                  >
                    <td className='p-3'>
                      <div className='flex items-center gap-2 text-sm text-gray-600'>
                        <Calendar className='w-4 h-4' />
                        {new Date(completion.checkedAt).toLocaleString()}
                      </div>
                    </td>
                    <td className='p-3'>
                      <Link
                        href={`/tasks/${completion.task.id}`}
                        className='text-blue-600 hover:text-blue-800 font-medium'
                      >
                        {completion.task.title}
                      </Link>
                    </td>
                    <td className='p-3'>
                      <div className='flex items-center gap-2'>
                        <Wrench className='w-4 h-4 text-gray-400' />
                        <span className='font-medium text-gray-800'>
                          {completion.element.name}
                        </span>
                      </div>
                    </td>
                    <td className='p-3'>
                      <div className='flex items-center gap-2 text-sm text-gray-600'>
                        <MapPin className='w-4 h-4' />
                        <div>
                          <div>{completion.element.location.name}</div>
                          <div className='text-xs text-gray-500 flex items-center gap-1'>
                            <Building2 className='w-3 h-3' />
                            {completion.element.location.department.name}
                          </div>
                        </div>
                      </div>
                    </td>
                    <td className='p-3'>
                      <span
                        className={`px-2 py-1 rounded text-xs font-medium ${
                          completion.status === "OK"
                            ? "bg-green-100 text-green-800"
                            : "bg-red-100 text-red-800"
                        }`}
                      >
                        {completion.status === "OK" ? (
                          <span className='flex items-center gap-1'>
                            <CheckCircle2 className='w-3 h-3' />
                            {t("tasks.history.ok") || "OK"}
                          </span>
                        ) : (
                          <span className='flex items-center gap-1'>
                            <XCircle className='w-3 h-3' />
                            {t("tasks.history.notOk") || "NOT OK"}
                          </span>
                        )}
                      </span>
                    </td>
                    <td className='p-3'>
                      {completion.notes ? (
                        <div className='max-w-xs p-2 bg-gray-50 rounded text-sm text-gray-700'>
                          <p className='whitespace-pre-wrap'>
                            {completion.notes}
                          </p>
                        </div>
                      ) : (
                        <span className='text-gray-400 text-sm'>{t("tasks.detail.noNotes") || "No notes added yet"}</span>
                      )}
                    </td>
                    <td className='p-3'>
                      {completion.ticket ? (
                        <Link
                          href={`/tickets/${completion.ticket.id}`}
                          className='text-blue-600 hover:text-blue-800 text-sm font-medium flex items-center gap-1'
                        >
                          <FileText className='w-3 h-3' />
                          {completion.ticket.title}
                        </Link>
                      ) : (
                        <span className='text-gray-400 text-sm'>—</span>
                      )}
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
          <div className='bg-gray-50 px-4 py-3 border-t'>
            <p className='text-sm text-gray-600'>
              Showing {completions.length} completion
              {completions.length !== 1 ? "s" : ""}
            </p>
          </div>
        </div>
      )}
    </main>
  );
}
